// Spline Cut.js ( Paste Spline.js )
//
// 2012-09-29: firset release
// 2012-10-05: changed script name. bug fixed. new functions added.
// 2013-07-13: bug fixed when illegal path exists.
//
// Hiroto Tsubaki tg@tres-graficos.jp
//

function buildUI( tool ) {
	tool.addParameterSeparator("Spline Cut");
	
	tool.addParameterLink("cut spline", false);
	
	tool.addParameterSelector("direction coord.", ["object", "world"], false, false);
	tool.addParameterSelector("cut direction", ["x", "y", "z"], false, false);
	
	tool.addParameterBool("only selected polygon", 0, 0, 1, false, false);
	
	tool.addParameterBool("point detection", 1, 0, 1, false, false);
	tool.addParameterBool("edge detection", 1, 0, 1, false, false);
	tool.addParameterBool("connect edge point", 1, 0, 1, false, false);
	
	tool.addParameterButton("cut", "apply", "cutSpline");
	
}

function cutSpline( tool ) {
	var doc = tool.document();
	var obj = doc.selectedObject();
	var spline = tool.getParameter("cut spline");
	
	if (!obj || !spline) {
		OS.beep(); return;
	}
	
	if (obj.type() == POLYGONOBJ && spline.family() == SPLINEFAMILY) {
		var pi, ei, i, j, k;
		
		var core = obj.core();
		var objMat = obj.obj2WorldMatrix();
		var objMatInverse = objMat.inverse();
		
		var splineCore = spline.core();
		var splineRot = spline.getParameter("rotation");
		var splineRotMat = new Mat4D( ROTATE_HPB, splineRot.x, splineRot.y, splineRot.z );
		var splineMat = spline.obj2WorldMatrix();
		
		//print('----');
		
		var dir = tool.getParameter("cut direction");
		var coord = tool.getParameter("direction coord.");
		
		var polysel = tool.getParameter("only selected polygon");
		
		var detectPoint = tool.getParameter("point detection");
		var detectEdge = tool.getParameter("edge detection");
		var tryConnect = tool.getParameter("connect edge point");
		
		var dirVec;
		switch( parseInt(dir) ) {
			case 0: // -x
				dirVec = new Vec3D( -1, 0, 0 );
				break;
			case 1: // -y
				dirVec = new Vec3D( 0, -1, 0 );
				break;
			case 2: // -z
				dirVec = new Vec3D( 0, 0, -1 );
				break;
		}
		// spline vector
		if (coord == 0) {
			dirVec = splineRotMat.multiply( dirVec );
		}
		dirVec = Vec3D_normalize( dirVec );
		
		var points = new Array();
		var edges = new Array();
		var pathCount = splineCore.pathCount();
		for (i = 0;i < pathCount;i++) {
			var cache = splineCore.cache( i );
			
			// error check.
			if (cache === undefined ) break;
			
			var cacheLen = cache.length;
			for (j = 0;j < cacheLen;j++) {
				points.pushUnique( splineMat.multiply( cache[j] ) );
			}
			for (j = 0;j < cacheLen-1;j++) {
				edges.push( [ splineMat.multiply( cache[j] ), splineMat.multiply( cache[j + 1] ) ] );
			}
		}
		// no spline points
		if (points.length < 1) {
			return 0;
		}
		
		var polygonCount = core.polygonCount();
		
		// for undo/redo
		if (tool.parameterWithName) obj.recordGeometryForUndo();
		
		var crossCountTotal = 0;
		var len = points.length;
		var edgesLen = edges.length;
		
		var raySize = 1;
		
		if (detectPoint) { // point detection
			for (i = 0;i < polygonCount;i++) {
				if (!polysel || core.polygonSelection( i ) ) {
					var polygonSize = core.polygonSize( i );
					var crossCount = 0;
				
					for (j = 0;j < polygonSize-2;j++) {
						var triangles = core.triangle(i, j);
						var t0 = core.vertex( core.vertexIndex( i, triangles[0] ) );
						var t1 = core.vertex( core.vertexIndex( i, triangles[1] ) );
						var t2 = core.vertex( core.vertexIndex( i, triangles[2] ) );
							
						t0 = objMat.multiply( t0 ); t1 = objMat.multiply( t1 ); t2 = objMat.multiply( t2 );

						for (pi = 0;pi < len;pi++) {
							var point = points[pi];
							
							var crossPoint = triangleIntersect( point, dirVec, t0, t1, t2 );
							
							if (crossPoint) {
								var vindex = core.addVertex( true, objMatInverse.multiply( crossPoint ) );
								core.setVertexSelection( vindex, true );
								crossCount++;
								//print( "crossPoint:" + crossPoint.x.toFixed(2) + ', ' + crossPoint.y.toFixed(2) + ', ' + crossPoint.z.toFixed(2) );
								
								//print( 'spline_point:'+vindex );
							}
						}
						crossCountTotal += crossCount;
					}
				}
			}
		}
		
		var vertexCount = core.vertexCount();
		if (detectEdge && vertexCount > 0) { // edge detection
			var maxVec = core.vertex(0);
			var minVec = core.vertex(0);
			for (i = 1;i < vertexCount;i++) { // calculate bounding box
				var vec = core.vertex( i );
				maxVec.x = Math.max( vec.x, maxVec.x ); maxVec.y = Math.max( vec.y, maxVec.y ); maxVec.z = Math.max( vec.z, maxVec.z );
				minVec.x = Math.min( vec.x, minVec.x ); minVec.y = Math.min( vec.y, minVec.y ); minVec.z = Math.min( vec.z, minVec.z );
			}
	//print( "minVec:" + minVec.x.toFixed(2) + ', ' + minVec.y.toFixed(2) + ', ' + minVec.z.toFixed(2) );
	//print( "maxVec:" + maxVec.x.toFixed(2) + ', ' + maxVec.y.toFixed(2) + ', ' + maxVec.z.toFixed(2) );
			
			var polygons = new Array();
			
			// collect edge
			var polyEdges = new Array();
			for (i = 0;i < polygonCount;i++) {
				if (!polysel || core.polygonSelection( i ) ) {
					var polygonSize = core.polygonSize( i );
					for (j = 0;j < polygonSize;j++) {
						var e0 = j;
						var e1 = (j == polygonSize-1)? 0 : j+1;
						
						var u0 = core.uvCoord( i, e0 );
						var u1 = core.uvCoord( i, e1 );
						
						var vi0 = core.vertexIndex( i, e0 );
						var vi1 = core.vertexIndex( i, e1 );
						
						polyEdges.pushEdge( new Edge( i, vi0, vi1, u0, u1, [vi0, vi1] ) );
					}
				}
			}
			//print( '====' );
			//print( polyEdges.length + ' edges detected.' );
			//

			var polyEdgesLen = polyEdges.length;
			var objSize = Vec3D_distance( minVec, maxVec );
			for (i = 0;i < polyEdgesLen;i++) { // scan all polygon edges
				var polyEdge = polyEdges[i];
				var crossCount = 0;
				
				var e0 = objMat.multiply( core.vertex( polyEdge.e0 ) );
				var e1 = objMat.multiply( core.vertex( polyEdge.e1 ) );
				
				var eDist = Vec3D_distance( e0, e1 );
				var eDirVec = Vec3D_normalize( e0.sub( e1 ) );
				
				for (ei = 0;ei < edgesLen;ei++) { // scan all spline edges
					var edge = edges[ei];
					var dist = Vec3D_distance( e0, edge[0] ) + objSize;
					
					var p0 = edge[0].add( dirVec.multiply( dist ) ); var p1 = edge[1].add( dirVec.multiply( dist ) );
					var p2 = edge[0].sub( dirVec.multiply( dist ) ); var p3 = edge[1].sub( dirVec.multiply( dist ) );
					
					// debugging
					//core.addPolygon( 3, false, [p0, p1, p2] ); core.addPolygon( 3, false, [p3, p2, p1] );
					
					var crossPoint = triangleIntersect( e0, eDirVec, p0, p1, p2 );
					if (!crossPoint) crossPoint = triangleIntersect( e0, eDirVec, p3, p2, p1 );
					
					if (crossPoint) {
						if ( Vec3D_distance( e0, crossPoint ) < eDist 
									&& Vec3D_distance( e1, crossPoint) < eDist ) {
							var vindex = core.addVertex( true, objMatInverse.multiply( crossPoint ) );
							core.setVertexSelection( vindex, true );
							crossCount++;
							
							//print( ' -- ' );
							//print( 'get crosspoint for edge '+ i );
							if (tryConnect) {
								for (j = 0;j < polyEdge.weight;j++) {
									var pindex = polyEdge.plist[j];
									var order = polyEdge.orders[j];
									//print( 'edge:'+ i+', for '+pindex+' in '+polyEdge.plist );
									
									var uv0 = polyEdge.ulist[j][0];
									var uv1 = polyEdge.ulist[j][1];
									
									if (polyEdge.e0 == polyEdge.orders[j][1]) {
  									var proportion = Vec3D_distance( e0, crossPoint ) / eDist;
  								} else {
  								  var proportion = Vec3D_distance( e1, crossPoint ) / eDist;
  								}
  								
									var uv = uv0.multiply( proportion ).add( uv1.multiply( 1 - proportion ) );
									
									//print( 'for '+pindex+', '+order+' with '+vindex );
									
									var pushed = false;
									var plen = polygons.length;
									for (k =0;k < plen;k++) {
										if ( polygons[k].refindex == pindex ) {
											pushed = true;
											var poly = polygons[k];
											//
											//print( k + ':' + poly );
											//print( 'search for '+vindex + ' with ' + order[0] + ' and ' + order[1] );
											var ilen = poly.indices.length;
											for (ii = 0;ii < ilen;ii++) {
												if ( poly.indices[ii] == order[0] ) {
  												// if polygon is regenerated already, detect order with added point(s).
  												var len0 = Vec3D_distance( core.vertex( order[0] ), core.vertex( vindex ) );
  												while ( ii < ilen ) {
  												  ii++;
  												  if (ii == ilen) {
  												    poly.indices.push( vindex );
  												    poly.uvs.push( uv );
  												  } else {
    												  var len1 = Vec3D_distance( core.vertex( order[0] ), core.vertex( poly.indices[ii] ) );
    												  if (len0 < len1) {
                                poly.indices.splice(ii, 0, vindex);
                                poly.uvs.splice(ii, 0, uv);
                                ii = ilen;
    												  }
    												}
  												}
												}
											}
											//print( k + ':' + poly );
											k = plen - 1;
										}
									}
									if (!pushed) {
										var polygonSize = core.polygonSize( pindex );
										var indices = [];
										var uvs = [];
										for (k = 0;k < polygonSize;k++) {
											indices.push( core.vertexIndex( pindex, k ) )
											uvs.push( core.uvCoord( pindex, k ) );
											
											if ( core.vertexIndex( pindex, k ) == order[0]) {
												indices.push( vindex );
												uvs.push( uv );
											}
										}
									  var poly = new Poly( pindex, indices, uvs );
										polygons.push( poly );
									  //print( 'added new polygon for polygon ' + pindex );
										//print( poly );
									}
								}
							}
						}
					}
				}
				crossCountTotal += crossCount;
			}
			
			//print( polygons.length + ' polygon(s) re-generated' );
			//print( polygons );
			var dels = [];
			if (tryConnect) {
				var plen = polygons.length;
				for (var pi = 0;pi < plen;pi++) {
					var poly = polygons[pi];
					var pindex = core.addIndexPolygon( poly.indices.length, poly.indices);
					var ulen = poly.uvs.length;
					for (var ui =0;ui < ulen;ui++) {
						core.setUVCoord( pindex, ui, poly.uvs[ ui ] );
					}
					dels.push( poly.refindex );
				}
			}
			if (dels.length > 0) {
				dels.sort();
				for (i = dels.length - 1;i >= 0;i--) {
					core.deletePolygon( dels[i] );
					//print( 'del:' + dels[i] );
				}
			}
		}
		
		print( crossCountTotal + ' point(s) added.' );
		
		obj.update();
	}
}

// Tomas Möller 
var triangleIntersect = function( orig, dir, v0, v1, v2 ) {
	var e1, e2, pvec, tvec, qvec;
	var epsilon = 1e-12;
	var det;
	var t, u, v;
	var inv_det;
	
	e1 = v1.sub( v0 );
	e2 = v2.sub( v0 );
	
	pvec = dir.cross( e2 );
	det = e1.dot( pvec );
	det = (det.dot)? det.x : det;

	if (det > epsilon) {
		tvec = orig.sub( v0 );
		u = tvec.dot( pvec );
		u = (u.dot)? u.x : u;
		if (u < 0 || u > det) return false;
		
		qvec = tvec.cross( e1 );
		
		v = dir.dot( qvec );
		v = (v.dot)? v.x : v;
		if (v < 0 || u + v > det) return false; 
	} else if (det < -epsilon) {
		tvec = orig.sub( v0 );
		
		u = tvec.dot( pvec );
		u = (u.dot)? u.x : u;
		if (u > 0 || u < det) return false;
		
		qvec = tvec.cross( e1 );
		
		v = dir.dot( qvec );
		v = (v.dot)? v.x : v;
		if (v > 0 || u + v < det) return false;
	} else {
		return false;
	}
	
	inv_det = 1 / det;
	
	t = e2.dot( qvec );
	t = (t.dot)? t.x : t;
	t = t * inv_det;
	
	u = u * inv_det;
	v = v * inv_det;

	var crossPoint = orig.add( dir.multiply( t ) );
	
	return crossPoint;
}

var Vec3D_distance = function() {
	if( arguments.length == 1)
			return Math.sqrt( arguments[0].x*arguments[0].x + arguments[0].y*arguments[0].y + arguments[0].z*arguments[0].z );
	var p = arguments[1].sub(arguments[0]);
	return Math.sqrt( p.x*p.x + p.y*p.y + p.z*p.z );
}

var Vec3D_normalize = function(vec) {
	var l = vec.norm();
	if (l != 0) {
		return vec.multiply( 1/l );
	}
	return vec;
}

var Poly = function( refindex, indices, uvs ) {
	this.refindex = refindex;
	this.indices = indices;
	this.uvs = uvs;
}

Poly.prototype.toString = function() {
  return this.refindex + ' [' + this.indices.length + '] i:' + this.indices;
}

var Edge = function( p0, e0, e1, u0, u1, order ) {
	this.e0 = e0;
	this.e1 = e1;
	
	this.plist = [ p0 ];
	this.ulist = [ [ u0, u1 ] ];
	this.orders = [ order ];
	
	this.weight = 1;
}

Edge.prototype.isEqualEdge = function( edge ) {
	return ((this.e0 == edge.e0 && this.e1 == edge.e1) || (this.e0 == edge.e1 && this.e1 == edge.e0));
}

Edge.prototype.toString = function() {
	return 'p:('+this.plist+') i:('+this.e0+', '+this.e1+') o:('+this.orders+') w:'+this.weight;
}

Array.prototype.pushUnique = function( val ) {
	var len = this.length;
	for (var i = 0;i < len;i++) {
		var data = this[i];
		if (data == val) {
			return false;
		}
	}
	this.push( val );
}

Array.prototype.toString = function() {
	var len = this.length;
	var str = '[';
	for (var i =0;i < len;i++) {
		str += this[i]
		if (i != len-1) str += ', ';
	}
	return str + ']';
}

Array.prototype.pushEdge = function( edge ) {
	var len = this.length;
	for (var i = 0;i < len;i++) {
		var e = this[i];
		if (e.isEqualEdge( edge )) {
			e.weight += 1;
			e.plist.push( edge.plist[0] );
			e.ulist.push( edge.ulist[0] );
			e.orders.push( edge.orders[0] );
			
			return i;
		}
	}
	this.push( edge );
	return -1;
}